'use strict';

/* --------------------------------------------------------------
 swiper.js 2020-06-02
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/* globals Swiper */

/**
 * Widget that binds the swiper plugin (third party) to a DOM element
 *
 * @todo Remove the try - catch blocks and and correct the swiper issues.
 */
gambio.widgets.module('swiper', [gambio.source + '/libs/events', gambio.source + '/libs/responsive'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $body = $('body'),
        $slides = null,
        $controls = null,
        $target = null,
        $theme = null,
        init = true,
        swiper = null,
        sliderOptions = null,
        hasThumbnails = true,
        mode = null,
        breakpointDataset = null,
        duplicates = false,
        preventSlideStart = false,
        sliderDefaults = {
        observer: true,
        // Default configuration for the swiper
        pagination: '.swiper-pagination',
        nextButton: '.swiper-button-next',
        prevButton: '.swiper-button-prev',
        paginationClickable: true,
        loop: true,
        autoplay: 3,
        autoplayDisableOnInteraction: false
    },
        defaults = {
        observer: true,
        // JSON that gets merged with the sliderDefaults and is passed to "swiper" directly.
        sliderOptions: null,
        // If this instance is a "main" swiper, the given selector selects the "control" swiper.
        controls: null,
        // If this instance is a "control" swiper, the given selector selects the "main" swiper.
        target: null,
        // Sets the initial slide (needed to prevent different init slides in main/controller slider).
        initSlide: null,
        // Detect if a swiper is needed for the breakpoint. If not, turn it off
        autoOff: false,
        // The translucence fix enables support for a fade effect between images with different aspect ratio,
        // but causing a delay between the change
        disableTranslucenceFix: false,
        breakpoints: [{
            // Until which breakpoint this settings is available
            breakpoint: 40,
            // If true, the paging bullets will be replaced with the preview images.
            usePreviewBullets: false,
            // This and all other settings belonging to the swiper plugin.
            slidesPerView: 2,
            // If true, the current slide gets centered in view (most usefull with an even slidesPerView
            // count).
            centeredSlides: true
        }, {
            breakpoint: 60,
            usePreviewBullets: true,
            slidesPerView: 3
        }, {
            breakpoint: 80,
            usePreviewBullets: true,
            slidesPerView: 3
        }, {
            breakpoint: 100,
            usePreviewBullets: true,
            slidesPerView: 5
        }]
    },
        options = $.extend({}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Function that generates the markup for
     * the preview bullets
     * @param       {Swiper}        swiper          Swiper object
     * @param       {integer}       index           Index of the slide
     * @param       {string}        className       The classname that must be add to the markup
     * @return      {string}                        The preview image html string
     * @private
     */
    var _generatePreviewButtons = function _generatePreviewButtons(swiper, index, className) {
        var $currentSlide = $slides.eq(index),
            $image = $currentSlide.find('img'),
            altTxt = $image.attr('alt'),
            thumbImage = $currentSlide.data('thumbImage');

        if (thumbImage) {
            return '<img src="' + thumbImage + '" alt="' + altTxt + '" class="' + className + '" />';
        }

        return '';
    };

    /**
     * Helper function to get the index of the
     * active slide
     * @return     {integer}                       The index of the active slide
     * @private
     */
    var _getIndex = function _getIndex() {
        var index = $this.find('.swiper-slide-active').index();

        // If there are duplicate slides (generated
        // by the swiper) recalculate the index
        index = duplicates ? index - 1 : index;
        index = index || 0;

        return index;
    };

    /**
     * Helper function to add the active
     * class to the active slide
     * @param       {integer}           index       The index of the active slide
     * @private
     */
    var _setActive = function _setActive(index) {
        $slides = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');
        index = duplicates ? index + 1 : index;
        $slides.removeClass('active').eq(index).addClass('active');
    };

    // ########## EVENT HANDLER ##########

    /**
     * Event handler for the mouseenter event.
     * It disables the autoplay
     * @private
     */
    var _mouseEnterHandler = function _mouseEnterHandler() {
        try {
            if (swiper) {
                swiper.stopAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the mouseleave event.
     * It enables the autoplay
     * @private
     */
    var _mouseLeaveHandler = function _mouseLeaveHandler() {
        try {
            if (swiper) {
                swiper.startAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the goto event.
     * It switches the current slide to the given index
     * and adds the active class to the new active slide
     * @param       {object}    e       jQuery event object
     * @param       {number}    d       Index of the slide to show
     * @private
     */
    var _gotoHandler = function _gotoHandler(e, d) {
        e.stopPropagation();

        // Set the active slide
        _setActive(d);

        // Temporary deactivate the onSlideChangeStart event
        // to prevent looping through the goto / changeStart
        // events
        preventSlideStart = true;

        // Remove the autoplay after a goto event
        $this.off('mouseleave.swiper');
        swiper.stopAutoplay();

        // Try to correct the index between sliders
        // with and without duplicates
        var index = duplicates ? d + 1 : d;
        if (index > $slides.length) {
            index = 0;
        }

        // Goto the desired slide
        swiper.slideTo(index);

        // Reactivate the onSlideChangeEvent
        preventSlideStart = false;
    };

    /**
     * Click event handler that triggers a
     * "goto" event to the target swiper
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _clickHandler = function _clickHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var $self = $(this),
            index = $self.index();

        index = duplicates ? index - 1 : index;

        // Set the active slide
        _setActive(index);

        // Inform the main swiper
        $target.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
    };

    /**
     * Event that gets triggered on slideChange.
     * If the slide gets changed, the controls
     * will follow the current slide in position
     * @private
     */
    var _triggerSlideChange = function _triggerSlideChange() {
        if (!preventSlideStart) {
            var index = _getIndex(),
                lastIndex = $slides.closest(".swiper-slide-prev").index() || $slides.length - 2;

            var previousSlider = $slides.closest(".swiper-slide-prev");
            if (previousSlider) {
                lastIndex = duplicates ? previousSlider.index() - 1 : previousSlider.index();
            }

            if (index > $slides.length - 1) {
                index = 0;
            }

            // Recalculate index if duplicate slides are inside the slider
            if (index < 0) {
                index = $slides.length + index;
            } else {
                index = duplicates && index === lastIndex ? index - lastIndex : index;
            }

            // Set the active slide
            _setActive(index);

            // Inform the controls
            $controls.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
        }
    };

    /**
     * Workaround for the translucence issue
     * that happens on small screens with enabled
     * fade effect. Maybe it can be removed, if the
     * swiper gets updated itself
     * @private
     */
    var _translucenceWorkaround = function _translucenceWorkaround() {
        if (!options.disableTranslucenceFix && sliderOptions && sliderOptions.effect === 'fade') {
            $this.find('.swiper-slide').filter(':not(.swiper-slide-active)').fadeTo(300, 0, function () {
                $(this).css('visibility', 'hidden');
            });

            $this.find('.swiper-slide').filter('.swiper-slide-active').fadeTo(300, 1, function () {
                $(this).css('visibility', '');
            });
        }
    };

    /**
     * The breakpoint handler initializes the swiper
     * with the settings for the current breakpoint.
     * Therefore it uses the default slider options,
     * the custom slider options given by the options
     * object and the breakpoint options object also
     * given by the options (in this order)
     * @private
     */
    var _breakpointHandler = function _breakpointHandler() {

        // Get the current viewmode
        var oldMode = mode || {},
            newMode = jse.libs.theme.responsive.breakpoint(),
            extendOptions = options.breakpoints[0] || {},
            newBreakpointDataset = null;

        // Only do something if the view was changed
        if (newMode.id !== oldMode.id) {

            // Store the new viewmode
            mode = $.extend({}, newMode);

            // Iterate through the breakpoints object to detect
            // the correct settings for the current breakpoint
            $.each(options.breakpoints, function (i, v) {
                if (v.breakpoint > newMode.id) {
                    return false;
                }
                newBreakpointDataset = i;
                extendOptions = v;
            });

            if (options.sliderOptions && options.sliderOptions.breakpoints) {
                $.each(options.sliderOptions.breakpoints, function (i, v) {
                    if (v.breakpoint === newMode.id) {
                        extendOptions = v;
                        return false;
                    }
                });
            }

            // Only do something if the settings change due browser
            // resize or if it's the first time run
            if (newBreakpointDataset !== breakpointDataset || init) {
                // Combine the settings
                sliderOptions = $.extend({}, sliderDefaults, options.sliderOptions || {}, extendOptions);

                // Add the preview image bullets function to the options object
                if (sliderOptions.usePreviewBullets && hasThumbnails) {
                    sliderOptions.paginationBulletRender = _generatePreviewButtons;
                }

                // Add the autoplay interval to the options object
                sliderOptions.autoplay = sliderOptions.autoplay ? sliderOptions.autoplay * 1000 : 0;

                // Disable loop if there is only one slider. 
                if ($this.find('.swiper-slide').length === 1) {
                    sliderOptions.loop = false;
                }

                // If an swiper exists, get the current
                // slide no. and remove the old swiper
                if (swiper) {
                    sliderOptions.initialSlide = _getIndex();
                    try {
                        swiper.destroy(true, true);
                    } catch (ignore) {
                        swiper = null;
                    }
                } else {
                    sliderOptions.initialSlide = options.initSlide || sliderOptions.initialSlide || 0;
                }

                var $duplicate = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');

                if (!options.autoOff || $duplicate.length > sliderOptions.slidesPerView && options.autoOff) {
                    $this.addClass('swiper-is-active').removeClass('swiper-is-not-active');

                    // Initialize the swiper
                    try {
                        swiper = new Swiper($this, sliderOptions);
                    } catch (e) {
                        return; // Swiper might throw an error upon initialization that should not halt the script execution.
                    }

                    swiper.off('onTransitionEnd onSlideChangeStart').on('onTransitionEnd', _translucenceWorkaround);

                    // If this is a "main" swiper and has external controls, an
                    // goto event is triggered if the current slide is changed
                    if ($controls.length) {
                        swiper.on('onSlideChangeStart', _triggerSlideChange);
                    }

                    // Add the event handler
                    $this.off('mouseenter.swiper mouseleave.swiper ' + jse.libs.theme.events.SWIPER_GOTO() + ' ' + jse.libs.theme.events.SLIDES_UPDATE()).on('mouseenter.swiper', _mouseEnterHandler).on('mouseleave.swiper', _mouseLeaveHandler).on(jse.libs.theme.events.SWIPER_GOTO(), _gotoHandler).on(jse.libs.theme.events.SLIDES_UPDATE(), _updateSlides);

                    if (init) {
                        // Check if there are duplicates slides (generated by the swiper)
                        // after the first time init of the swiper
                        duplicates = !!$this.find('.swiper-slide-duplicate').length;
                    }

                    // Set the active slide
                    var index = init && options.initSlide ? options.initSlide : _getIndex();
                    _setActive(index);

                    // Inform the controls that the main swiper has changed
                    // In case that the other slider isn't initialized yet,
                    // set an data attribute to the markup element to inform
                    // it on init
                    if ($controls.length) {
                        $controls.attr('data-swiper-init-slide', index);
                        _triggerSlideChange();
                    }

                    // Unset the init flag
                    init = false;

                    setTimeout(function () {
                        return swiper.update();
                    });
                } else {
                    // Disable the swiper buttons
                    $this.removeClass('swiper-is-active').addClass('swiper-is-not-active');
                    init = true;
                }
            }
        }
    };

    /**
     * Event handler that adds & removes slides from the
     * swiper. After the slides were processed, the first
     * slide is shown
     * @param       {object}    e       jQuery event object
     * @param       {object}    d       JSON object that contains the categories / images
     * @private
     */
    var _updateSlides = function _updateSlides(e, d) {

        // Loops through each category inside the images array
        $.each(d, function (category, dataset) {
            var catName = category + '-category',
                add = [],
                remove = [],
                markup = $theme.html();

            // Get all indexes from the slides
            // of the same category and remove
            // them from the slider
            $slides.filter('.' + catName).each(function () {
                var $self = $(this),
                    index = $self.data().swiperSlideIndex;

                index = index === undefined ? $self.index() : index;
                remove.push(index);
            });
            swiper.removeSlide(remove);

            // Generate the markup for the new slides
            // and add them to the slider
            $.each(dataset || [], function (i, v) {
                v.className = catName;
                v.srcattr = 'src="' + v.src + '"';
                add.push(Mustache.render(markup, v));
            });
            swiper.appendSlide(add);
        });

        $slides = $this.find('.swiper-slide');

        // To prevent an inconsistent state
        // in control / main slider combinations
        // slide to the first slide
        _setActive(0);
        var index = duplicates ? 1 : 0;
        swiper.slideTo(index, 0);
    };

    /**
     * Prevent text selection by clicking on swiper buttons
     * @private
     */
    var _preventTextSelection = function _preventTextSelection() {
        $(options.sliderOptions.nextButton).on('selectstart', function () {
            return false;
        });
        $(options.sliderOptions.prevButton).on('selectstart', function () {
            return false;
        });
    };

    /**
     * Sets the initial height for one swiper image container to prevent cut off images on smaller swiper heights
     * @private
     */
    var _scaleThumbnailHeight = function _scaleThumbnailHeight() {
        var swiperContainer = $('.swiper-container-vertical .swiper-slide');
        var $containerHeight = swiperContainer.css('height');

        // Workaround for IE Browsers
        if ($('.swiper-container-vertical').hasClass('swiper-wp8-vertical')) {
            $containerHeight = swiperContainer.height() + 5;

            swiperContainer.css('height', $containerHeight);
        }

        if ($containerHeight === '0px') {
            $containerHeight = $('.product-info-thumbnails-mobile').css('height');
        }

        $('.align-middle').css('height', $containerHeight);
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        $slides = $this.find('.swiper-slide');
        $controls = $(options.controls);
        $target = $(options.target);
        $theme = $this.find('theme');

        // Check if all images inside the swiper have
        // thumbnail image given
        $slides.each(function () {
            if (!$(this).data().thumbImage) {
                hasThumbnails = false;
                return false;
            }
        });

        // Add the breakpoint handler ty dynamically
        // set the options corresponding to the browser size (slider responsive will re-initialize the swiper).
        _breakpointHandler();

        // If this instance is a "control" swiper the target is the main swiper
        // which will be updated on a click inside this control swiper
        if (options.target) {
            $this.on('click.swiper', '.swiper-slide', _clickHandler);
        }

        $(document).ready(function () {
            $('.swiper-vertical .swiper-slide[data-index]').css('display', 'inline-block');
            $('.product-info-image .swiper-slide[data-index]').css('z-index', 'inherit');
            $('.product-info-image .swiper-slide[data-index] .swiper-slide-inside img.img-responsive').fadeIn(1000);
        });

        _translucenceWorkaround();
        _preventTextSelection();
        _scaleThumbnailHeight();

        // Fix for invisible Thumbnail-Images for switching from Tablet-Portrait to Tablet-Landscape
        $body.on(jse.libs.theme.events.BREAKPOINT(), function () {
            _scaleThumbnailHeight();
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
